<?php
// Include database connection and mail functionality
require 'dbconfig/dbconfig.php';
require 'MailFuntion/Mails.php';

// Initialize variables for order status and error message
$order_success = false;
$error_message = '';

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Validate and sanitize form data
    $first_name = filter_var(trim($_POST['first_name']), FILTER_SANITIZE_STRING);
    $last_name = filter_var(trim($_POST['last_name']), FILTER_SANITIZE_STRING);
    $address_line1 = filter_var(trim($_POST['address_line1']), FILTER_SANITIZE_STRING);
    $address_line2 = filter_var(trim($_POST['address_line2'] ?? ''), FILTER_SANITIZE_STRING);
    $city = filter_var(trim($_POST['city']), FILTER_SANITIZE_STRING);
    $postal_code = filter_var(trim($_POST['postal_code'] ?? ''), FILTER_SANITIZE_STRING);
    $contact_number = filter_var(trim($_POST['contact_number']), FILTER_SANITIZE_STRING);
    $email = filter_var(trim($_POST['email']), FILTER_SANITIZE_EMAIL);
    $cart_data = json_decode($_POST['cart_data'], true);
    $total_price = floatval($_POST['total_price']);
    $payment_method = 'cash';

    // Validate required fields
    if (empty($first_name) || empty($last_name) || empty($address_line1) || empty($city) || empty($contact_number) || empty($email) || !is_array($cart_data) || $total_price <= 0) {
        $error_message = 'Please fill in all required fields and ensure valid cart data.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error_message = 'Invalid email address.';
    } elseif ($city === 'Other' && empty($address_line2)) {
        $error_message = 'Please enter your city in the additional address field when selecting "Other".';
    } else {
        // Verify database connection
        if (!$conn) {
            $error_message = 'Database connection failed: ' . mysqli_connect_error();
            error_log('Database connection failed: ' . mysqli_connect_error(), 3, 'errors.log');
        } else {
            // Begin transaction to ensure data consistency
            $conn->begin_transaction();

            try {
                // Prepare and bind SQL statement for orders table
                $stmt = $conn->prepare("INSERT INTO orders (first_name, last_name, address_line1, address_line2, city, postal_code, contact_number, email, total_price, payment_method, order_date) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
                if (!$stmt) {
                    throw new Exception('Prepare failed for orders: ' . $conn->error);
                }

                $stmt->bind_param("ssssssssds", $first_name, $last_name, $address_line1, $address_line2, $city, $postal_code, $contact_number, $email, $total_price, $payment_method);

                // Execute the statement for orders table
                if (!$stmt->execute()) {
                    throw new Exception('Execute failed for orders: ' . $stmt->error);
                }

                // Get the generated order_id
                $order_id = $conn->insert_id;

                // Prepare and bind SQL statement for order_details table
                $stmt_details = $conn->prepare("INSERT INTO order_details (order_id, item_name, quantity, price) VALUES (?, ?, ?, ?)");
                if (!$stmt_details) {
                    throw new Exception('Prepare failed for order_details: ' . $conn->error);
                }

                // Insert each cart item into order_details
                foreach ($cart_data as $item) {
                    $item_name = filter_var(trim($item['name']), FILTER_SANITIZE_STRING);
                    $quantity = intval($item['quantity']);
                    $price = floatval($item['price']);
                    $stmt_details->bind_param("isid", $order_id, $item_name, $quantity, $price);
                    if (!$stmt_details->execute()) {
                        throw new Exception('Execute failed for order_details: ' . $stmt_details->error);
                    }
                }

                // Commit transaction
                $conn->commit();
                $order_success = true;

                // Send order confirmation email
                try {
                    sendOrderConfirmationEmail($first_name, $last_name, $email, $cart_data, $total_price, $address_line1, $address_line2, $city, $postal_code, $contact_number, $order_id);
                } catch (Exception $e) {
                    error_log('Email sending failed: ' . $e->getMessage(), 3, 'errors.log');
                    // Note: Email failure does not affect order_success
                }

            } catch (Exception $e) {
                // Rollback transaction on error
                $conn->rollback();
                $error_message = 'Database error: ' . $e->getMessage();
                error_log('Transaction failed: ' . $e->getMessage(), 3, 'errors.log');
            }

            // Close statements
            if (isset($stmt)) {
                $stmt->close();
            }
            if (isset($stmt_details)) {
                $stmt_details->close();
            }
        }

        // Close database connection
        $conn->close();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <!-- Meta tags for character encoding and responsive viewport -->
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customer Details</title>
    <!-- Google Fonts for Roboto -->
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <!-- Font Awesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Inline CSS for styling -->
    <style>
        body {
            font-family: 'Roboto', sans-serif;
            background-color: #f4f4f4;
            margin: 0;
            padding: 20px;
        }

        .customer-container {
            max-width: 1100px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }

        h2, h3 {
            color: #333;
            text-align: center;
        }

        .form-container {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
        }

        .customer-form {
            flex: 1;
            /*min-width: 300px;*/
            margin-right:20px;
        }
        
        .order-summary {
            flex: 1;
            /*min-width: 300px;*/
        }

        #order-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 15px;
        }

        #order-table th, #order-table td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }

        #order-table th {
            background-color: #f4f4f4;
            font-weight: 500;
            color: #333;
        }

        #order-table td {
            color: #555;
        }

        #order-table tr:hover {
            background-color: #f9f9f9;
        }

        .total {
            font-size: 1.2em;
            font-weight: 500;
            text-align: right;
            color: #333;
        }

        .customer-form input {
            width: 100%;
            padding: 10px;
            margin: 10px 0;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 1em;
        }
        
        .customer-form select {
            width: 100%;
            padding: 10px;
            margin: 10px 0;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 1em;
        }

        .customer-form input:disabled {
            background-color: #f0f0f0;
            cursor: not-allowed;
        }

        .customer-form button {
            width: 100%;
            padding: 12px;
            background-color: #28a745;
            color: white;
            border: none;
            border-radius: 4px;
            font-size: 1.1em;
            cursor: pointer;
            transition: background-color 0.3s;
        }

        .customer-form button:hover:not(:disabled) {
            background-color: #218838;
        }

        .customer-form button:disabled {
            background-color: #6c757d;
            cursor: not-allowed;
        }

        .error-message {
            display: none;
            padding: 15px;
            margin: 10px 0;
            border-radius: 4px;
            text-align: center;
            background-color: #f8d7da;
            color: #721c24;
        }

        .error-message.show {
            display: block;
        }

        .home-button {
            display: flex;
            align-items: center;
            justify-content: center;
            position: fixed;
            top: 1rem;
            left: 1rem;
            width: 3rem;
            height: 3rem;
            background-color: #2ecc71;
            border-radius: 50%;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
            transition: all 0.3s ease;
            z-index: 1000;
        }

        .home-button:hover {
            background-color: #27ae60;
            transform: scale(1.1);
        }

        .home-button img {
            width: 1.5rem;
            height: 1.5rem;
            object-fit: contain;
        }

        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }

        .modal.show {
            display: flex;
        }

        .modal-content {
            background-color: white;
            padding: 30px;
            border-radius: 12px;
            text-align: center;
            max-width: 400px;
            width: 90%;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.2);
            animation: slideIn 0.3s ease-out;
        }

        .modal-content .success-icon {
            font-size: 3rem;
            color: #28a745;
            margin-bottom: 20px;
        }

        .modal-content h2 {
            margin: 0 0 15px;
            font-size: 1.5rem;
            color: #333;
        }

        .modal-content p {
            margin: 0 0 20px;
            color: #555;
            font-size: 1rem;
        }

        .modal-content button {
            padding: 10px 20px;
            background-color: #28a745;
            color: white;
            border: none;
            border-radius: 4px;
            font-size: 1rem;
            cursor: pointer;
            transition: background-color 0.3s;
        }

        .modal-content button:hover {
            background-color: #218838;
        }

        @keyframes slideIn {
            from {
                transform: translateY(-50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }

        @media (max-width: 768px) {
            .home-button {
                width: 2.5rem;
                height: 2.5rem;
            }

            .home-button img {
                width: 1.2rem;
                height: 1.2rem;
            }

            .modal-content {
                padding: 20px;
            }
        }

        @media (max-width: 480px) {
            .home-button {
                width: 2rem;
                height: 2rem;
            }

            .home-button img {
                width: 1rem;
                height: 1rem;
            }

            .modal-content {
                padding: 15px;
            }

            .modal-content h2 {
                font-size: 1.3rem;
            }

            .modal-content p {
                font-size: 0.9rem;
            }
        }
    </style>
</head>
<body class="bg-gray-100 font-roboto">
    <!-- Home button to return to index.html -->
    <a href="index.html" class="home-button">
        <img src="assets/home.png" alt="Home" class="home-icon">
    </a>

    <!-- Success Modal -->
    <div id="success-modal" class="modal <?php echo $order_success ? 'show' : ''; ?>">
        <div class="modal-content">
            <i class="fas fa-check-circle success-icon"></i>
            <h2>Order Placed Successfully!</h2>
            <p>Thank you for your order. You'll receive a confirmation email soon.</p>
            <button onclick="window.location.href='index.html'">Return to Home</button>
        </div>
    </div>

    <!-- Error message for failed submission -->
    <div id="error-message" class="error-message <?php echo $error_message ? 'show' : ''; ?>">
        <?php echo htmlspecialchars($error_message); ?>
    </div>

    <!-- Customer details container -->
    <div class="customer-container">
        <h2>Customer Details</h2>
        <div class="form-container">
            <!-- Order summary section -->
            <div class="order-summary">
                <h3>Order Summary</h3>
                <table id="order-table">
                    <thead>
                        <tr>
                            <th>Item</th>
                            <th>Quantity</th>
                            <th>Price (LKR)</th>
                        </tr>
                    </thead>
                    <tbody id="order-summary"></tbody>
                </table>
                <p class="total">Total: LKR <span id="total-price">0.00</span></p>
            </div>

            <!-- Customer form section -->
            <div class="customer-form">
                <form action="" method="POST" id="order-form">
                    <input type="text" name="first_name" placeholder="First Name*" required>
                    <input type="text" name="last_name" placeholder="Last Name*" required>
                    <input type="text" name="address_line1" placeholder="Address: 123, Galle Road, Colombo*" required>
                    <select name="city" id="city-select" required>
                        <option value="" disabled selected>Select City*</option>
                        <option value="Akarawita">Akarawita</option>
                        <option value="Akkaraipattu">Akkaraipattu</option>
                        <option value="Aluthgama">Aluthgama</option>
                        <option value="Ambalangoda">Ambalangoda</option>
                        <option value="Ambalantota">Ambalantota</option>
                        <option value="Ampara">Ampara</option>
                        <option value="Ampitiya">Ampitiya</option>
                        <option value="Anuradhapura">Anuradhapura</option>
                        <option value="Athurugiriya">Athurugiriya</option>
                        <option value="Avissawella">Avissawella</option>
                        <option value="Badulla">Badulla</option>
                        <option value="Balangoda">Balangoda</option>
                        <option value="Bambalapitiya">Bambalapitiya</option>
                        <option value="Bandarawela">Bandarawela</option>
                        <option value="Batawala">Batawala</option>
                        <option value="Battaramulla">Battaramulla</option>
                        <option value="Batticaloa">Batticaloa</option>
                        <option value="Batugampola">Batugampola</option>
                        <option value="Batuwatta">Batuwatta</option>
                        <option value="Bentota">Bentota</option>
                        <option value="Beruwala">Beruwala</option>
                        <option value="Bibile">Bibile</option>
                        <option value="Bollete">Bollete</option>
                        <option value="Bope">Bope</option>
                        <option value="Boralesgamuwa">Boralesgamuwa</option>
                        <option value="Borella">Borella</option>
                        <option value="Chavakachcheri">Chavakachcheri</option>
                        <option value="Chilaw">Chilaw</option>
                        <option value="Chunnakam">Chunnakam</option>
                        <option value="Colombo">Colombo</option>
                        <option value="Colpetty">Colpetty</option>
                        <option value="Dambulla">Dambulla</option>
                        <option value="Dankotuwa">Dankotuwa</option>
                        <option value="Dedigamuwa">Dedigamuwa</option>
                        <option value="Dehiwala-Mount Lavinia">Dehiwala-Mount Lavinia</option>
                        <option value="Deltara">Deltara</option>
                        <option value="Dikwella">Dikwella</option>
                        <option value="Ella">Ella</option>
                        <option value="Elpitiya">Elpitiya</option>
                        <option value="Embilipitiya">Embilipitiya</option>
                        <option value="Eravur">Eravur</option>
                        <option value="Fort">Fort</option>
                        <option value="Galle">Galle</option>
                        <option value="Galewela">Galewela</option>
                        <option value="Gampaha">Gampaha</option>
                        <option value="Gampola">Gampola</option>
                        <option value="Ganemulla">Ganemulla</option>
                        <option value="Gonapinuwala">Gonapinuwala</option>
                        <option value="Habarakada">Habarakada</option>
                        <option value="Hambantota">Hambantota</option>
                        <option value="Handapangoda">Handapangoda</option>
                        <option value="Hanwella">Hanwella</option>
                        <option value="Haputale">Haputale</option>
                        <option value="Hatton">Hatton</option>
                        <option value="Havelock Town">Havelock Town</option>
                        <option value="Hewainna">Hewainna</option>
                        <option value="Hikkaduwa">Hikkaduwa</option>
                        <option value="Hiripitiya">Hiripitiya</option>
                        <option value="Hokandara">Hokandara</option>
                        <option value="Homagama">Homage</option>
                        <option value="Horagama">Horagala</option>
                        <option value="Horana">Horana</option>
                        <option value="Jaffna">Jaffna</option>
                        <option value="Ja-Ela">Ja-Ela</option>
                        <option value="Kaduwela">Kaduwela</option>
                        <option value="Kahawala">Kahawala</option>
                        <option value="Kalmunai">Kalmunai</option>
                        <option value="Kalatuwawa">Kalatuwawa</option>
                        <option value="Kalutara">Kalutara</option>
                        <option value="Kandana">Kandana</option>
                        <option value="Kandy">Kandy</option>
                        <option value="Kapugoda">Kapugoda</option>
                        <option value="Katana">Katana</option>
                        <option value="Kataragama">Kataragama</option>
                        <option value="Kattankudy">Kattankudy</option>
                        <option value="Katunayake">Katunayake</option>
                        <option value="Kehelwatta">Kehelwatta</option>
                        <option value="Kegalle">Kegalle</option>
                        <option value="Kekirawa">Kekirawa</option>
                        <option value="Kesbewa">Kesbewa</option>
                        <option value="Kilinochchi">Kilinochchi</option>
                        <option value="Kinniya">Kinniya</option>
                        <option value="Kiriwattuduwa">Kiriwattuduwa</option>
                        <option value="Kirulapone">Kirulapone</option>
                        <option value="Kochchikade">Kochchikade</option>
                        <option value="Kolonnawa">Kolonnawa</option>
                        <option value="Kosgama">Kosgama</option>
                        <option value="Kottegoda">Kottegoda</option>
                        <option value="Kuliyapitiya">Kuliyapitiya</option>
                        <option value="Kurunegala">Kurunegala</option>
                        <option value="Madapatha">Madapatha</option>
                        <option value="Maharagama">Maharagama</option>
                        <option value="Mahiyangana">Mahiyangana</option>
                        <option value="Malabe">Malabe</option>
                        <option value="Mannar">Mannar</option>
                        <option value="Marawila">Marawila</option>
                        <option value="Matale">Matale</option>
                        <option value="Matara">Matara</option>
                        <option value="Mawanella">Mawanella</option>
                        <option value="Medirigiriya">Medirigiriya</option>
                        <option value="Mihintale">Mihintale</option>
                        <option value="Minuwangoda">Minuwangoda</option>
                        <option value="Monaragala">Monaragala</option>
                        <option value="Moratuwa">Moratuwa</option>
                        <option value="Mount Lavinia">Mount Lavinia</option>
                        <option value="Mullegama">Mullegama</option>
                        <option value="Mullaitivu">Mullaitivu</option>
                        <option value="Mulleriyawa New Town">Mulleriyawa New Town</option>
                        <option value="Mutur">Mutur</option>
                        <option value="Mutwal">Mutwal</option>
                        <option value="Nallur">Nallur</option>
                        <option value="Nanaddan">Nanaddan</option>
                        <option value="Napawela">Napawela</option>
                        <option value="Narammala">Narammala</option>
                        <option value="Nawalapitiya">Nawalapitiya</option>
                        <option value="Nedunkeni">Nedunkeni</option>
                        <option value="Negombo">Negombo</option>
                        <option value="Nugegoda">Nugegoda</option>
                        <option value="Nuwara Eliya">Nuwara Eliya</option>
                        <option value="Oddamavadi">Oddamavadi</option>
                        <option value="Padukka">Padukka</option>
                        <option value="Pallai">Pallai</option>
                        <option value="Panadura">Panadura</option>
                        <option value="Pannala">Pannala</option>
                        <option value="Pannipitiya">Pannipitiya</option>
                        <option value="Paranthan">Paranthan</option>
                        <option value="Peradeniya">Peradeniya</option>
                        <option value="Pitipana Homagama">Pitipana Homagama</option>
                        <option value="Point Pedro">Point Pedro</option>
                        <option value="Polgasowita">Polgasowita</option>
                        <option value="Polonnaruwa">Polonnaruwa</option>
                        <option value="Pottuvil">Pottuvil</option>
                        <option value="Pugoda">Pugoda</option>
                        <option value="Puthukkudiyiruppu">Puthukkudiyiruppu</option>
                        <option value="Puttalam">Puttalam</option>
                        <option value="Puwakpitiya">Puwakpitiya</option>
                        <option value="Ragama">Ragama</option>
                        <option value="Rambukkana">Rambukkana</option>
                        <option value="Ranala">Ranala</option>
                        <option value="Ratnapura">Ratnapura</option>
                        <option value="Sainthamaruthu">Sainthamaruthu</option>
                        <option value="Sammanthurai">Sammanthurai</option>
                        <option value="Siddamulla">Siddamulla</option>
                        <option value="Slave Island">Slave Island</option>
                        <option value="Sri Jayawardenepura">Sri Jayawardenepura</option>
                        <option value="Talaimannar">Talaimannar</option>
                        <option value="Talawakele">Talawakele</option>
                        <option value="Talawatugoda">Talawatugoda</option>
                        <option value="Tangalle">Tangalle</option>
                        <option value="Tissamaharama">Tissamaharama</option>
                        <option value="Tittapattara">Tittapattara</option>
                        <option value="Trincomalee">Trincomalee</option>
                        <option value="Tummodara">Tummodara</option>
                        <option value="Undugoda">Undugoda</option>
                        <option value="Valaichchenai">Valaichchenai</option>
                        <option value="Valvettithurai">Valvettithurai</option>
                        <option value="Vavuniya">Vavuniya</option>
                        <option value="Waga">Waga</option>
                        <option value="Wahacotte">Wahacotte</option>
                        <option value="Walpola">Walpola</option>
                        <option value="Wariyapola">Wariyapola</option>
                        <option value="Watareka">Watareka</option>
                        <option value="Wattala">Wattala</option>
                        <option value="Weligama">Weligama</option>
                        <option value="Wellawatta">Wellawatta</option>
                        <option value="Wellawaya">Wellawaya</option>
                        <option value="Wennappuwa">Wennappuwa</option>
                        <option value="Other" style="font-weight: bold;">Other</option>
                    </select>
                    <input type="text" name="address_line2" id="address-line2" placeholder="Enter city if 'Other' is selected" disabled>
                    <input type="text" name="postal_code" placeholder="Postal Code">
                    <input type="text" name="contact_number" placeholder="Contact Number*" maxlength="10" required>
                    <input type="email" name="email" placeholder="Email Address*" required>
                    <h4>Cash On Delivery Only</h4>
                    <h2>FREE DELIVERY</h2>
                    <input type="hidden" name="payment_method" value="cash">
                    <input type="hidden" name="cart_data" id="cart-data">
                    <input type="hidden" name="total_price" id="hidden-total-price">
                    <button type="submit" id="place-order-btn">Place the Order</button>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Function to get query parameters from URL
        function getQueryParam(param) {
            const urlParams = new URLSearchParams(window.location.search);
            return urlParams.get(param);
        }

        // Load and display cart data on page load
        document.addEventListener('DOMContentLoaded', () => {
            const cartData = getQueryParam('cart');
            const orderForm = document.getElementById('order-form');
            const placeOrderBtn = document.getElementById('place-order-btn');
            const successModal = document.getElementById('success-modal');
            const citySelect = document.getElementById('city-select');
            const addressLine2 = document.getElementById('address-line2');
            const errorMessage = document.getElementById('error-message');

            // Handle city selection to enable/disable address_line2
            citySelect.addEventListener('change', () => {
                if (citySelect.value === 'Other') {
                    addressLine2.disabled = false;
                    addressLine2.required = true;
                } else {
                    addressLine2.disabled = true;
                    addressLine2.required = false;
                    addressLine2.value = ''; // Clear the field when disabled
                }
            });

            // Form validation before submission
            orderForm.addEventListener('submit', (event) => {
                if (citySelect.value === 'Other' && !addressLine2.value.trim()) {
                    event.preventDefault();
                    errorMessage.textContent = 'Please enter your city in the additional address field when selecting "Other".';
                    errorMessage.classList.add('show');
                    placeOrderBtn.disabled = false;
                    placeOrderBtn.textContent = 'Place the Order';
                    return;
                }
                placeOrderBtn.disabled = true;
                placeOrderBtn.textContent = 'Processing...';
            });

            if (cartData) {
                try {
                    const cart = JSON.parse(decodeURIComponent(cartData));
                    const orderSummary = document.getElementById('order-summary');
                    let total = 0;

                    // Validate cart data
                    if (!Array.isArray(cart) || cart.length === 0) {
                        throw new Error('Cart is empty or invalid.');
                    }

                    // Populate table rows
                    cart.forEach(item => {
                        if (!item.name || !item.quantity || !item.price) {
                            throw new Error('Invalid item data in cart.');
                        }
                        orderSummary.innerHTML += `
                            <tr>
                                <td>${item.name}</td>
                                <td>${item.quantity}</td>
                                <td>${(item.price * item.quantity).toFixed(2)}</td>
                            </tr>
                        `;
                        total += item.price * item.quantity;
                    });

                    // Update total price and hidden form fields
                    document.getElementById('total-price').textContent = total.toFixed(2);
                    document.getElementById('cart-data').value = JSON.stringify(cart);
                    document.getElementById('hidden-total-price').value = total.toFixed(2);
                } catch (e) {
                    errorMessage.textContent = 'Invalid cart data: ' + e.message;
                    errorMessage.classList.add('show');
                    console.error('Cart data parsing error:', e);
                }
            } else {
                errorMessage.textContent = 'No cart data provided.';
                errorMessage.classList.add('show');
            }

            // Handle order success or failure
            const orderSuccess = <?php echo json_encode($order_success); ?>;
            if (orderSuccess) {
                // Keep button disabled after success
                placeOrderBtn.disabled = true;
                placeOrderBtn.textContent = 'Order Placed';
                successModal.classList.add('show');
                // Auto-redirect after 5 seconds
                setTimeout(() => {
                    window.location.href = 'index.html';
                }, 5000);
            } else if (<?php echo json_encode($error_message); ?>) {
                // Re-enable button on error
                placeOrderBtn.disabled = false;
                placeOrderBtn.textContent = 'Place the Order';
                console.log('Order failed:', <?php echo json_encode($error_message); ?>);
            }
        });
    </script>
</body>
</html>